// Copyright 2018-2024 Emil Dotchevski and Reverge Studios, Inc.
// Distributed under the Boost Software License, Version 1.0. (See accompanying
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifdef BOOST_LEAF_TEST_SINGLE_HEADER
#   include "leaf.hpp"
#else
#   include <boost/leaf/result.hpp>
#   include <boost/leaf/handle_errors.hpp>
#endif

#include "_test_res.hpp"
#include "lightweight_test.hpp"
#ifdef BOOST_LEAF_BOOST_AVAILABLE
#   include <boost/config/workaround.hpp>
#else
#   define BOOST_WORKAROUND(a,b) 0
#endif

namespace leaf = boost::leaf;

struct value
{
    int x;

    explicit value( int x_ ): x(x_) { };

#ifndef BOOST_LEAF_NO_CXX11_REF_QUALIFIERS
    value( value const & ) = delete;
    value( value && ) = default;
#endif
};

leaf::result<value> f1()
{
    return value { 21 };
}

leaf::result<value> f2()
{
    BOOST_LEAF_ASSIGN(auto a, f1());
#if BOOST_WORKAROUND( BOOST_GCC, < 50000 ) || BOOST_WORKAROUND( BOOST_CLANG, <= 30800 )
    return std::move(a); // Older compilers are confused, but...
#else
    return a; // ...this doesn't need to be return std::move(a);
#endif
}

leaf::result<value> f3()
{
    BOOST_LEAF_ASSIGN(auto a, f2());
    BOOST_LEAF_ASSIGN(auto b, f2()); // Invoking the macro twice in the same scope, testing the temp name generation
    return value { a.x + b.x };
}

int main()
{
    BOOST_TEST_EQ(f3()->x, 42);

    {
        int r = leaf::try_handle_all(
            []() -> leaf::result<int>
            {
                int x = 42;

                leaf::result<int> r1(x);
                BOOST_LEAF_ASSIGN(auto && rx1, r1);
                BOOST_TEST_EQ(r1.value(), rx1);

                leaf::result<int &> r2(x);
                BOOST_LEAF_ASSIGN(auto && rx2, r2);
                BOOST_TEST_EQ(r2.value(), rx2);

                leaf::result<int &> r3(x);
                BOOST_LEAF_ASSIGN(auto & rx3, r3);
                BOOST_TEST_EQ(&r3.value(), &rx3);

                return 0;
            },
            []
            {
                return 1;
            } );
        BOOST_TEST_EQ(r, 0);
    }

#ifndef BOOST_LEAF_NO_CXX11_REF_QUALIFIERS
    {
        auto r = []() -> test_res<int, test_error>
        {
            int v = 0;
            BOOST_LEAF_ASSIGN(v, ([]() -> test_res<int, test_error> { return test_error(42); }()));
            return v;
        }();
        BOOST_TEST(!r);
        BOOST_TEST(r.error().moved);
        BOOST_TEST_EQ(r.error().value, 42);
    }
#endif

    return boost::report_errors();
}
