/* Example of Outcome try used with foreign object
(C) 2019-2026 Niall Douglas <http://www.nedproductions.biz/> (1 commit)


Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "../../../include/boost/outcome.hpp"

#include <iostream>

namespace outcome = BOOST_OUTCOME_V2_NAMESPACE;

//! [foreign_type]
enum Errc
{
  kBadValue
};
template <class T, class E = Errc> struct ForeignExpected
{
  T Value;
  E Error;
  int IsErrored;

  ForeignExpected(T v)
      : Value(v)
      , Error()
      , IsErrored(0)
  {
  }
  ForeignExpected(E e)
      : Value()
      , Error(e)
      , IsErrored(1)
  {
  }
};
//! [foreign_type]

//! [tell_outcome]
BOOST_OUTCOME_V2_NAMESPACE_BEGIN
template <class T, class E>  //
inline bool try_operation_has_value(const ForeignExpected<T, E> &v)
{
  return 0 == v.IsErrored;
}
template <class T, class E>  //
inline auto try_operation_return_as(const ForeignExpected<T, E> &v)
{
  switch(v.Error)
  {
  case kBadValue:
    return failure(make_error_code(std::errc::argument_out_of_domain));
  }
  abort();
}
template <class T, class E>  //
inline auto try_operation_extract_value(const ForeignExpected<T, E> &v)
{
  return v.Value;
}
BOOST_OUTCOME_V2_NAMESPACE_END
//! [tell_outcome]

//! [functions]
ForeignExpected<int> old_code(int a)  // old code
{
  if(0 == a)
    return kBadValue;
  return a;
}

outcome::result<int> new_code(int a)  // new code
{
  BOOST_OUTCOME_TRY(auto x, old_code(a));
  return x;
}
//! [functions]

int main()
{
  //! [example]
  auto printresult = [](const char *desc, auto x) {
    if(x)
    {
      std::cout << desc << " returns successful " << x.value() << std::endl;
    }
    else
    {
      std::cout << desc << " returns failure " << x.error().message() << std::endl;
    }
  };
  printresult("\nnew_code(5)", new_code(5));
  printresult("\nnew_code(0)", new_code(0));
  //! [example]
  return 0;
}
