// Copyright 2017, 2021, 2022 Peter Dimov.
// Distributed under the Boost Software License, Version 1.0.
// https://www.boost.org/LICENSE_1_0.txt

#include <boost/system/result.hpp>
#include <boost/core/lightweight_test.hpp>
#include <boost/core/lightweight_test_trait.hpp>
#include <boost/config.hpp>
#include <system_error>

using namespace boost::system;

struct X
{
    int v_;

    explicit X( int v ): v_( v ) {}

    X( X const& ) = delete;
    X& operator=( X const& ) = delete;
};

struct Y
{
};

struct E
{
};

BOOST_NORETURN void throw_exception_from_error( Y const &, boost::source_location const& )
{
    throw E();
}

struct E2
{
};

int main()
{
    {
        result<int> r;

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value(), 0 );
        BOOST_TEST_EQ( *r, 0 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        result<int> const r;

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value(), 0 );
        BOOST_TEST_EQ( *r, 0 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        BOOST_TEST( result<int>().has_value() );
        BOOST_TEST( !result<int>().has_error() );

        BOOST_TEST( result<int>() );
        BOOST_TEST_NOT( !result<int>() );

        BOOST_TEST_EQ( result<int>().value(), 0 );
        BOOST_TEST_EQ( *result<int>(), 0 );

        BOOST_TEST( result<int>().operator->() != 0 );
    }

    {
        result<int> r( 1 );

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value(), 1 );
        BOOST_TEST_EQ( *r, 1 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        result<int> const r( 1 );

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value(), 1 );
        BOOST_TEST_EQ( *r, 1 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        BOOST_TEST( result<int>( 1 ).has_value() );
        BOOST_TEST( !result<int>( 1 ).has_error() );

        BOOST_TEST( result<int>( 1 ) );
        BOOST_TEST_NOT( !result<int>( 1 ) );

        BOOST_TEST_EQ( result<int>( 1 ).value(), 1 );
        BOOST_TEST_EQ( *result<int>( 1 ), 1 );

        BOOST_TEST( result<int>( 1 ).operator->() != 0 );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        result<int> r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        result<int> const r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        BOOST_TEST( !result<int>( ec ).has_value() );
        BOOST_TEST( result<int>( ec ).has_error() );

        BOOST_TEST_NOT( result<int>( ec ) );
        BOOST_TEST( !result<int>( ec ) );

        BOOST_TEST_THROWS( result<int>( ec ).value(), system_error );

        BOOST_TEST_THROWS( *result<int>( ec ), system_error );
        BOOST_TEST_THROWS( result<int>( ec ).operator->(), system_error );
    }

    {
        auto ec = make_error_code( std::errc::invalid_argument );

        result<int, std::error_code> const r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::system_error );

        BOOST_TEST_THROWS( *r, std::system_error );
        BOOST_TEST_THROWS( r.operator->(), std::system_error );
    }

    {
        result<int, errc::errc_t> const r( in_place_error, errc::invalid_argument );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        result<int, std::errc> const r( std::errc::invalid_argument );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::system_error );

        BOOST_TEST_THROWS( *r, std::system_error );
        BOOST_TEST_THROWS( r.operator->(), std::system_error );
    }

    {
        result<int, std::exception_ptr> const r( std::make_exception_ptr( E2() ) );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

#if defined(BOOST_CLANG_VERSION) && BOOST_CLANG_VERSION < 30600
#else
        BOOST_TEST_THROWS( r.value(), E2 );
        BOOST_TEST_THROWS( *r, E2 );
        BOOST_TEST_THROWS( r.operator->(), E2 );
#endif
    }

    {
        result<int, std::exception_ptr> const r( in_place_error );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::bad_exception );

        BOOST_TEST_THROWS( *r, std::bad_exception );
        BOOST_TEST_THROWS( r.operator->(), std::bad_exception );
    }

    {
        result<X> r( 1 );

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value().v_, 1 );
        BOOST_TEST_EQ( (*r).v_, 1 );
        BOOST_TEST_EQ( r->v_, 1 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        result<X> const r( 1 );

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value().v_, 1 );
        BOOST_TEST_EQ( (*r).v_, 1 );
        BOOST_TEST_EQ( r->v_, 1 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        BOOST_TEST( result<X>( 1 ).has_value() );
        BOOST_TEST( !result<X>( 1 ).has_error() );

        BOOST_TEST( result<X>( 1 ) );
        BOOST_TEST_NOT( !result<X>( 1 ) );

        BOOST_TEST_EQ( result<X>( 1 ).value().v_, 1 );
        BOOST_TEST_EQ( (*result<X>( 1 )).v_, 1 );
        BOOST_TEST_EQ( result<X>( 1 )->v_, 1 );
    }

    {
        auto ec = Y();

        result<X, Y> r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), E );

        BOOST_TEST_THROWS( *r, E );
        BOOST_TEST_THROWS( r.operator->(), E );
    }

    {
        auto ec = Y();

        result<X, Y> const r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), E );

        BOOST_TEST_THROWS( *r, E );
        BOOST_TEST_THROWS( r.operator->(), E );
    }

    {
        auto ec = Y();

        BOOST_TEST(( !result<X, Y>( ec ).has_value() ));
        BOOST_TEST(( result<X, Y>( ec ).has_error() ));

        BOOST_TEST_NOT(( result<X, Y>( ec ) ));
        BOOST_TEST(( !result<X, Y>( ec ) ));

        BOOST_TEST_THROWS( (result<X, Y>( ec ).value()), E );

        BOOST_TEST_THROWS( *(result<X, Y>( ec )), E );
        BOOST_TEST_THROWS( (result<X, Y>( ec )).operator->(), E );
    }

    //

    {
        result<void> r;

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_NO_THROW( r.value() );

        BOOST_TEST_NO_THROW( *r );
        BOOST_TEST_NO_THROW( r.operator->() );

        BOOST_TEST( r.operator->() != 0 );
    }

    {
        result<void> const r;

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_NO_THROW( r.value() );

        BOOST_TEST_NO_THROW( *r );
        BOOST_TEST_NO_THROW( r.operator->() );

        BOOST_TEST( r.operator->() != 0 );
    }

    {
        BOOST_TEST( result<void>().has_value() );
        BOOST_TEST( !result<void>().has_error() );

        BOOST_TEST( result<void>() );
        BOOST_TEST_NOT( !result<void>() );

        BOOST_TEST_NO_THROW( result<void>().value() );

        BOOST_TEST_NO_THROW( *result<void>() );
        BOOST_TEST_NO_THROW( result<void>().operator->() );

        BOOST_TEST( result<void>().operator->() != 0 );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        result<void> r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        result<void> const r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        BOOST_TEST( !result<void>( ec ).has_value() );
        BOOST_TEST( result<void>( ec ).has_error() );

        BOOST_TEST_NOT( result<void>( ec ) );
        BOOST_TEST( !result<void>( ec ) );

        BOOST_TEST_THROWS( result<void>( ec ).value(), system_error );

        BOOST_TEST_THROWS( *result<void>( ec ), system_error );
        BOOST_TEST_THROWS( result<void>( ec ).operator->(), system_error );
    }

    {
        auto ec = make_error_code( std::errc::invalid_argument );

        result<void, std::error_code> const r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::system_error );

        BOOST_TEST_THROWS( *r, std::system_error );
        BOOST_TEST_THROWS( r.operator->(), std::system_error );
    }

    {
        result<void, errc::errc_t> const r( in_place_error, errc::invalid_argument );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        result<void, std::errc> const r( std::errc::invalid_argument );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::system_error );

        BOOST_TEST_THROWS( *r, std::system_error );
        BOOST_TEST_THROWS( r.operator->(), std::system_error );
    }

    {
        result<void, std::exception_ptr> const r( std::make_exception_ptr( E2() ) );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

#if defined(BOOST_CLANG_VERSION) && BOOST_CLANG_VERSION < 30600
#else
        BOOST_TEST_THROWS( r.value(), E2 );
        BOOST_TEST_THROWS( *r, E2 );
        BOOST_TEST_THROWS( r.operator->(), E2 );
#endif
    }

    {
        result<void, std::exception_ptr> const r( in_place_error );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::bad_exception );

        BOOST_TEST_THROWS( *r, std::bad_exception );
        BOOST_TEST_THROWS( r.operator->(), std::bad_exception );
    }

    //

    {
        int x1 = 1;

        result<int&> r( x1 );

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value(), 1 );
        BOOST_TEST_EQ( *r, 1 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        int x1 = 1;

        result<int&> const r( x1 );

        BOOST_TEST( r.has_value() );
        BOOST_TEST( !r.has_error() );

        BOOST_TEST( r );
        BOOST_TEST_NOT( !r );

        BOOST_TEST_EQ( r.value(), 1 );
        BOOST_TEST_EQ( *r, 1 );

        BOOST_TEST_EQ( r.operator->(), &*r );
    }

    {
        int x1 = 1;

        BOOST_TEST( result<int&>( x1 ).has_value() );
        BOOST_TEST( !result<int&>( x1 ).has_error() );

        BOOST_TEST( result<int&>( x1 ) );
        BOOST_TEST_NOT( !result<int&>( x1 ) );

        BOOST_TEST_EQ( result<int&>( x1 ).value(), 1 );
        BOOST_TEST_EQ( *result<int&>( x1 ), 1 );

        BOOST_TEST_EQ( result<int&>( x1 ).operator->(), &x1 );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        result<int&> r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        result<int&> const r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        auto ec = make_error_code( errc::invalid_argument );

        BOOST_TEST( !result<int&>( ec ).has_value() );
        BOOST_TEST( result<int&>( ec ).has_error() );

        BOOST_TEST_NOT( result<int&>( ec ) );
        BOOST_TEST( !result<int&>( ec ) );

        BOOST_TEST_THROWS( result<int&>( ec ).value(), system_error );

        BOOST_TEST_THROWS( *result<int&>( ec ), system_error );
        BOOST_TEST_THROWS( result<int&>( ec ).operator->(), system_error );
    }

    {
        auto ec = make_error_code( std::errc::invalid_argument );

        result<int&, std::error_code> const r( ec );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::system_error );

        BOOST_TEST_THROWS( *r, std::system_error );
        BOOST_TEST_THROWS( r.operator->(), std::system_error );
    }

    {
        result<int&, errc::errc_t> const r( in_place_error, errc::invalid_argument );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), system_error );

        BOOST_TEST_THROWS( *r, system_error );
        BOOST_TEST_THROWS( r.operator->(), system_error );
    }

    {
        result<int&, std::errc> const r( std::errc::invalid_argument );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::system_error );

        BOOST_TEST_THROWS( *r, std::system_error );
        BOOST_TEST_THROWS( r.operator->(), std::system_error );
    }

    {
        result<int&, std::exception_ptr> const r( std::make_exception_ptr( E2() ) );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

#if defined(BOOST_CLANG_VERSION) && BOOST_CLANG_VERSION < 30600
#else
        BOOST_TEST_THROWS( r.value(), E2 );
        BOOST_TEST_THROWS( *r, E2 );
        BOOST_TEST_THROWS( r.operator->(), E2 );
#endif
    }

    {
        result<int&, std::exception_ptr> const r( in_place_error );

        BOOST_TEST( !r.has_value() );
        BOOST_TEST( r.has_error() );

        BOOST_TEST_NOT( r );
        BOOST_TEST( !r );

        BOOST_TEST_THROWS( r.value(), std::bad_exception );

        BOOST_TEST_THROWS( *r, std::bad_exception );
        BOOST_TEST_THROWS( r.operator->(), std::bad_exception );
    }

    //

    return boost::report_errors();
}
